// Game_Music_Emu 0.4.0. http://www.slack.net/~ant/

#include "Nsfe_Emu.h"

#include "blargg_endian.h"
#include <string.h>

/* Copyright (C) 2005-2006 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for
more details. You should have received a copy of the GNU Lesser General
Public License along with this module; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */



#include "blargg_source.h"




static Music_Emu* new_nsfe_emu() { return BLARGG_NEW Nsfe_Emu; }

gme_type_t_ const gme_nsfe_type [1] = { &new_nsfe_emu, &Nsfe_Emu::read_info };

Nsfe_Info::Nsfe_Info()
{
	playlist_disabled = false;
	blargg_verify_byte_order();
}

Nsfe_Info::~Nsfe_Info() { }

// TODO: if no playlist, treat as if there is a playlist that is just 1,2,3,4,5... ?
void Nsfe_Info::disable_playlist( bool b )
{
	playlist_disabled = b;
	info_.track_count = playlist_size();
	if ( !info_.track_count || playlist_disabled )
		info_.track_count = track_count_;
}

int Nsfe_Info::remap_track( int i ) const
{
	if ( playlist_disabled || !playlist_size() )
		return i;
	
	return playlist_entry( i );
}

void Nsfe_Emu::start_track_( int i )
{
	Nsf_Emu::start_track_( remap_track( i ) );
}

const char* Nsfe_Info::track_name( int i ) const
{
	i = remap_track( i );
	if ( i < track_names.size() )
		return track_names [i];
	
	return "";
}

long Nsfe_Info::track_time( int i ) const
{
	i = remap_track( i );
	if ( i < track_times.size() )
		return track_times [i];
	
	return 0;
}

// Read little-endian 32-bit int
static blargg_err_t read_le32( Data_Reader& in, long* out )
{
	unsigned char buf [4];
	RETURN_ERR( in.read( buf, sizeof buf ) );
	*out = get_le32( buf );
	return 0;
}

// Read multiple strings and separate into individual strings
static blargg_err_t read_strs( Data_Reader& in, long size, blargg_vector<char>& chars,
		blargg_vector<const char*>& strs )
{
	RETURN_ERR( chars.resize( size + 1 ) );
	chars [size] = 0; // in case last string doesn't have terminator
	RETURN_ERR( in.read( &chars [0], size ) );
	
	RETURN_ERR( strs.resize( 128 ) );
	int count = 0;
	for ( int i = 0; i < size; i++ )
	{
		if ( strs.size() <= count )
			RETURN_ERR( strs.resize( count * 2 ) );
		strs [count++] = &chars [i];
		while ( i < size && chars [i] )
			i++;
	}
	
	return strs.resize( count );
}

// Copy in to out, where out has out_max characters allocated. Truncate to
// out_max - 1 characters.
static void copy_str( const char* in, char* out, int out_max )
{
	out [out_max - 1] = 0;
	strncpy( out, in, out_max - 1 );
}

struct nsfe_info_t {
	unsigned char load_addr [2];
	unsigned char init_addr [2];
	unsigned char play_addr [2];
	unsigned char speed_flags;
	unsigned char chip_flags;
	unsigned char track_count;
	unsigned char first_track;
};
BOOST_STATIC_ASSERT( sizeof (nsfe_info_t) == 10 );

blargg_err_t Nsfe_Info::load( const header_t& nsfe_tag, Data_Reader& in, void* loader_data,
		blargg_err_t (*loader)( void* data, Nsf_Emu::header_t const&, Data_Reader& ) )
{
	// check header
	if ( memcmp( nsfe_tag.tag, "NSFE", 4 ) )
		return "Not an NSFE file";
	
	// free previous info
	track_name_data.clear();
	track_names.clear();
	playlist.clear();
	track_times.clear();
	
	// default nsf header
	static const Nsf_Emu::header_t base_header =
	{
		{'N','E','S','M','\x1A'},// tag
		1,                  // version
		1, 1,               // track count, first track
		{0,0},{0,0},{0,0},  // addresses
		"","","",           // strings
		{0x1A, 0x41},       // NTSC rate
		{0,0,0,0,0,0,0,0},  // banks
		{0x20, 0x4E},       // PAL rate
		0, 0,               // flags
		{0,0,0,0}           // unused
	};
	Nsf_Emu::header_t& header = info_;
	header = base_header;
	
	// parse tags
	int phase = 0;
	while ( phase != 3 )
	{
		// read size and tag
		long size = 0;
		long tag = 0;
		RETURN_ERR( read_le32( in, &size ) );
		RETURN_ERR( read_le32( in, &tag ) );
		
		//dprintf( "tag: %c%c%c%c\n", char(tag), char(tag>>8), char(tag>>16), char(tag>>24) );
		
		switch ( tag )
		{
			case 'OFNI': {
				check( phase == 0 );
				if ( size < 8 )
					return "Bad NSFE file";
				
				nsfe_info_t info;
				info.track_count = 1;
				info.first_track = 0;
				
				int s = size;
				if ( s > (int) sizeof info )
					s = sizeof info;
				RETURN_ERR( in.read( &info, s ) );
				RETURN_ERR( in.skip( size - s ) );
				phase = 1;
				info_.speed_flags = info.speed_flags;
				info_.chip_flags = info.chip_flags;
				info_.track_count = info.track_count;
				this->track_count_ = info.track_count;
				info_.first_track = info.first_track;
				memcpy( info_.load_addr, info.load_addr, 2 * 3 );
				break;
			}
			
			case 'KNAB':
				if ( size > (int) sizeof info_.banks )
					return "Bad NSFE file";
				RETURN_ERR( in.read( info_.banks, size ) );
				break;
			
			case 'htua': {
				blargg_vector<char> chars;
				blargg_vector<const char*> strs;
				RETURN_ERR( read_strs( in, size, chars, strs ) );
				int n = strs.size();
				
				if ( n > 3 )
					copy_str( strs [3], info_.dumper, sizeof info_.dumper );
				
				if ( n > 2 )
					copy_str( strs [2], info_.copyright, sizeof info_.copyright );
				
				if ( n > 1 )
					copy_str( strs [1], info_.author, sizeof info_.author );
				
				if ( n > 0 )
					copy_str( strs [0], info_.game, sizeof info_.game );
				
				break;
			}
			
			case 'emit': {
				RETURN_ERR( track_times.resize( size / 4 ) );
				for ( int i = 0; i < track_times.size(); i++ )
					RETURN_ERR( read_le32( in, &track_times [i] ) );
				break;
			}
			
			case 'lblt':
				RETURN_ERR( read_strs( in, size, track_name_data, track_names ) );
				break;
			
			case 'tslp':
				RETURN_ERR( playlist.resize( size ) );
				RETURN_ERR( in.read( &playlist [0], size ) );
				break;
			
			case 'ATAD': {
				check( phase == 1 );
				phase = 2;
				if ( !loader )
				{
					in.skip( size );
				}
				else
				{
					Subset_Reader sub( &in, size ); // limit emu to nsf data
					RETURN_ERR( loader( loader_data, info_, sub ) );
					check( sub.remain() == 0 );
				}
				break;
			}
			
			case 'DNEN':
				check( phase == 2 );
				phase = 3;
				break;
			
			default:
				// tags that can be skipped start with a lowercase character
				check( std::islower( (tag >> 24) & 0xFF ) );
				RETURN_ERR( in.skip( size ) );
				break;
		}
	}
	
	disable_playlist( playlist_disabled );
	
	return 0;
}

static blargg_err_t nsf_loader( void* data, Nsf_Emu::header_t const& h, Data_Reader& in )
{
	return ((Nsf_Emu*) data)->load( h, in );
}

blargg_err_t Nsfe_Info::load( const header_t& h, Data_Reader& in, Nsf_Emu* emu )
{
	return load( h, in, emu, (emu ? nsf_loader : 0) );
}

blargg_err_t Nsfe_Info::load( Data_Reader& in, Nsf_Emu* nsf_emu )
{
	header_t h;
	RETURN_ERR( in.read( &h, sizeof h ) );
	return load( h, in, nsf_emu );
}

blargg_err_t Nsfe_Info::load_file( const char* path, Nsf_Emu* emu )
{
	Vfs_File_Reader in;
	RETURN_ERR( in.open( path ) );
	return load( in, emu );
}

blargg_err_t Nsfe_Info::track_info( track_info_t* out, int track ) const
{
	out->init();
	
	long length = track_time( track );
	if ( length > 0 )
		out->length = length;
	
	out->track_count = info_.track_count;
	out->copy_field( out->song, track_name( track ) );
	
	GME_COPY_FIELD( info_, out, game );
	GME_COPY_FIELD( info_, out, author );
	GME_COPY_FIELD( info_, out, copyright );
	GME_COPY_FIELD( info_, out, dumper );
	out->copy_field( out->system, "Nintendo NES" );
	return out->error();
}

blargg_err_t Nsfe_Emu::track_info( track_info_t* out, int track ) const
{
	return Nsfe_Info::track_info( out, track );
}

blargg_err_t Nsfe_Info::read_info( Data_Reader& in, track_info_t* out, int track )
{
	Nsfe_Info info;
	RETURN_ERR( info.load( in ) );
	return info.track_info( out, track );
}
